﻿# Script to find out if machine is Device Guard compliant
# requires driver verifier on system.
param([switch]$Capable, [switch]$Ready, [switch]$Enable, [switch]$Disable, $SIPolicyPath, [switch]$AutoReboot, [switch]$DG, [switch]$CG, [switch]$HVCI, [switch]$HLK, [switch]$Clear, [switch]$ResetVerifier)

$path = "C:\DGLogs\"
$LogFile = $path + "DeviceGuardCheckLog.txt"

$CompatibleModules = New-Object System.Text.StringBuilder
$FailingModules = New-Object System.Text.StringBuilder
$FailingExecuteWriteCheck = New-Object System.Text.StringBuilder

$DGVerifyCrit = New-Object System.Text.StringBuilder
$DGVerifyWarn = New-Object System.Text.StringBuilder
$DGVerifySuccess = New-Object System.Text.StringBuilder


$Sys32Path = "$env:windir\system32"
$DriverPath = "$env:windir\system32\drivers"

#generated by certutil -encode 
$SIPolicy_Encoded = "BQAAAA43RKLJRAZMtVH2AW5WMHbk9wcuTBkgTbfJb0SmxaI0BACNkAgAAAAAAAAA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"

$HSTITest_Encoded = "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"

function Log($message)
{
    $message | Out-File $LogFile -Append -Force
}

function LogAndConsole($message)
{
    Write-Host $message
    Log $message
}

function LogAndConsoleWarning($message)
{
    Write-Host $message -foregroundcolor "Yellow"
    Log $message
}

function LogAndConsoleSuccess($message)
{
    Write-Host $message -foregroundcolor "Green"
    Log $message
}

function LogAndConsoleError($message)
{
    Write-Host $message -foregroundcolor "Red" 
    Log $message
}

function IsExempted([System.IO.FileInfo] $item)
{
    $cert = (Get-AuthenticodeSignature $item.FullName).SignerCertificate
    if($cert.ToString().Contains("CN=Microsoft Windows, O=Microsoft Corporation, L=Redmond, S=Washington, C=US"))
    {
        Log $item.FullName + "MS Exempted"
        return 1
    }
    else
    {
        Log $item.FullName + "Not-exempted"
        Log $cert.ToString()
        return 0
    }
} 

function CheckExemption($_ModName)
{
    $mod1 = Get-ChildItem $Sys32Path $_ModName
    $mod2 = Get-ChildItem $DriverPath $_ModName
    if($mod1)
    {   
        Log "NonDriver module" + $mod1.FullName
        return IsExempted($mod1) 
    }
    elseif($mod2)
    {
        Log "Driver Module" + $mod2.FullName
        return IsExempted($mod2)
    }

}

function CheckFailedDriver($_ModName, $CIStats)
{
    Log "Module: " $_ModName.Trim()
    if(CheckExemption($_ModName.Trim()) - eq 1)
    {
        $CompatibleModules.AppendLine("Windows Signed: " + $_ModName.Trim()) | Out-Null
        return
    }
    $index = $CIStats.IndexOf("execute pool type count:".ToLower())
    if($index -eq -1)
    {
        return
    }
    $_tempStr = $CIStats.Substring($index)
    $Result = "PASS"
    $separator = "`r`n",""
    $option = [System.StringSplitOptions]::RemoveEmptyEntries
    $stats = $_tempStr.Split($separator,$option)
    Log $stats.Count

    $FailingStat = ""
    foreach( $stat in $stats)
    {
        $_t =$stat.Split(":")
        if($_t.Count -eq 2 -and $_t[1].trim() -ne "0")
        {
            $Result = "FAIL"
            $FailingStat = $stat
            break
        }
    }
    if($Result.Contains("PASS"))
    {
        $CompatibleModules.AppendLine($_ModName.Trim()) | Out-Null 
    }
    elseif($FailingStat.Trim().Contains("execute-write"))
    {
        $FailingExecuteWriteCheck.AppendLine("Module: "+ $_ModName.Trim() + "`r`n`tReason: " + $FailingStat.Trim() ) | Out-Null  
    }
    else
    {
        $FailingModules.AppendLine("Module: "+ $_ModName.Trim() + "`r`n`tReason: " + $FailingStat.Trim() ) | Out-Null 
    }
    Log "Result: " $Result
}

function ListCIStats($_ModName, $str1)
{
    $i1 = $str1.IndexOf("Code Integrity Statistics:".ToLower())
    if($i1 -eq -1 )
    {
        Log "String := " $str1
        Log "Warning! CI Stats are missing for " $_ModName
        return 
    }
    $temp_str1 = $str1.Substring($i1)
    $CIStats = $temp_str1.Substring(0).Trim()

    CheckFailedDriver $_ModName $CIStats
}

function ListDrivers($str)
{
    $_tempStr= $str

    $separator = "module:",""
    $option = [System.StringSplitOptions]::RemoveEmptyEntries
    $index1 = $_tempStr.IndexOf("MODULE:".ToLower())
    if($index1 -lt 0)
    {
        return
    }
    $_tempStr = $_tempStr.Substring($Index1)
    $_SplitStr = $_tempStr.Split($separator,$option)


    Log $_SplitStr.Count
    LogAndConsole "Verifying each module please wait .... "
    foreach($ModuleDetail in $_Splitstr)
    {
        #LogAndConsole $Module
        $Index2 = $ModuleDetail.IndexOf("(")
        if($Index2 -eq -1)
        {
            "Skipping .."
            continue
        }
        $ModName = $ModuleDetail.Substring(0,$Index2-1)
        Log "Driver: " $ModName
        Log "Processing module: " $ModName
        ListCIStats $ModName $ModuleDetail
    }

    $DriverScanCompletedMessage = "Completed scan. List of Compatible Modules can be found at " + $LogFile
    LogAndConsole $DriverScanCompletedMessage 

    if($FailingModules.Length -gt 0 -or $FailingExecuteWriteCheck.Length -gt 0 )
    {
        $WarningMessage = "Incompatible HVCI Kernel Driver Modules found"
        if($HLK)
        {
            LogAndConsoleError $WarningMessage
        }
        else 
        {
            LogAndConsoleWarning $WarningMessage
        }

        LogAndConsoleError $FailingExecuteWriteCheck.ToString()
        if($HLK)
        {
            LogAndConsoleError $FailingModules.ToString()
        }
        else
        {
            LogAndConsoleWarning $FailingModules.ToString()
        }
        if($FailingModules.Length -ne 0 -or $FailingExecuteWriteCheck.Length -ne 0 )
        {
            if($HLK)
            {
                $DGVerifyCrit.AppendLine($WarningMessage) | Out-Null
            }
            else
            {
                $DGVerifyWarn.AppendLine($WarningMessage) | Out-Null
            }
        }
    }
    else
    {
        LogAndConsoleSuccess "No Incompatible Drivers found"
    }
}

function ListSummary()
{
    if($DGVerifyCrit.Length -ne 0 )
    {
        LogAndConsoleError "Machine is not Device Guard / Credential Guard compatible because of the following:"
        LogAndConsoleError $DGVerifyCrit.ToString()
        LogAndConsoleWarning $DGVerifyWarn.ToString()
        if(!$HVCI -and !$DG)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "CG_Capable" /t REG_DWORD /d 0 /f '
        }
        if(!$CG)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "DG_Capable" /t REG_DWORD /d 0 /f '
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "HVCI_Capable" /t REG_DWORD /d 0 /f '
        }

    }
    elseif ($DGVerifyWarn.Length -ne 0 )
    {
        LogAndConsoleSuccess "Device Guard / Credential Guard can be enabled on this machine.`n"
        LogAndConsoleWarning "The following additional qualifications, if present, can enhance the security of Device Guard / Credential Guard on this system:"
        LogAndConsoleWarning $DGVerifyWarn.ToString()
        if(!$HVCI -and !$DG)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "CG_Capable" /t REG_DWORD /d 1 /f '
        }
        if(!$CG)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "DG_Capable" /t REG_DWORD /d 1 /f '
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "HVCI_Capable" /t REG_DWORD /d 1 /f '
        }
    }
    else
    {
        LogAndConsoleSuccess "Machine is Device Guard / Credential Guard Ready.`n" 
        if(!$HVCI -and !$DG)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "CG_Capable" /t REG_DWORD /d 2 /f '
        }
        if(!$CG)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "DG_Capable" /t REG_DWORD /d 2 /f '
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "HVCI_Capable" /t REG_DWORD /d 2 /f '
        }
    }
}


function Instantiate-Kernel32 {
    try 
    {
        Add-Type -TypeDefinition @"
        using System;
        using System.Diagnostics;
        using System.Runtime.InteropServices;
     
        public static class Kernel32
        {
            [DllImport("kernel32", SetLastError=true, CharSet = CharSet.Ansi)]
                public static extern IntPtr LoadLibrary(
                    [MarshalAs(UnmanagedType.LPStr)]string lpFileName);
             
            [DllImport("kernel32", CharSet=CharSet.Ansi, ExactSpelling=true, SetLastError=true)]
                public static extern IntPtr GetProcAddress(
                    IntPtr hModule,
                    string procName);
        }
     
"@
    }
    catch
    {
        Log $_.Exception.Message 
        LogAndConsole "Instantiate-Kernel32 failed"
    }
}

function Instantiate-HSTI {
    try 
    {
        Add-Type -TypeDefinition @"
        using System;
        using System.Diagnostics;
        using System.Runtime.InteropServices;
        using System.Net;
     
        public static class HstiTest3
        {
            [DllImport("hstitest.dll", CharSet = CharSet.Unicode)]
            public static extern int QueryHSTIdetails( 
                ref HstiOverallError pHstiOverallError, 
                [In, Out] HstiProviderErrorDuple[] pHstiProviderErrors,
                ref uint pHstiProviderErrorsCount,
                byte[] hstiPlatformSecurityBlob,
                ref uint pHstiPlatformSecurityBlobBytes);

            [DllImport("hstitest.dll", CharSet = CharSet.Unicode)]
            public static extern int QueryHSTI(ref bool Pass); 
                       
            [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
            public struct HstiProviderErrorDuple
            {
                internal uint protocolError;
                internal uint role;
                internal HstiProviderErrors providerError;
                [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 256)]
                internal string ID;
                [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 4096)]
                internal string ErrorString;
            }
    
            [FlagsAttribute]
            public enum HstiProviderErrors : int
            {
                None              = 0x00000000,
                VersionMismatch   = 0x00000001,
                RoleUnknown       = 0x00000002,
                RoleDuplicated    = 0x00000004,
                SecurityFeatureSizeMismatch   = 0x00000008,
                SizeTooSmall      = 0x00000010,
                VerifiedMoreThanImplemented   = 0x00000020,
                VerifiedNotMatchImplemented   = 0x00000040
            }

            [FlagsAttribute]
            public enum HstiOverallError : int
            {
                None                               = 0x00000000,
                RoleTooManyPlatformReference       = 0x00000001,
                RoleTooManyIbv                     = 0x00000002,
                RoleTooManyOem                     = 0x00000004,
                RoleTooManyOdm                     = 0x00000008,
                RoleMissingPlatformReference       = 0x00000010,
                VerifiedIncomplete                 = 0x00000020,
                ProtocolErrors                     = 0x00000040,
                BlobVersionMismatch                = 0x00000080,
                PlatformSecurityVersionMismatch    = 0x00000100,
                ProviderError                      = 0x00000200
            }    
           
        }
"@

        $LibHandle = [Kernel32]::LoadLibrary("C:\Windows\System32\hstitest.dll")
        $FuncHandle = [Kernel32]::GetProcAddress($LibHandle, "QueryHSTIdetails")
        $FuncHandle2 = [Kernel32]::GetProcAddress($LibHandle, "QueryHSTI")

        if ([System.IntPtr]::Size -eq 8) 
        {
            #assuming 64 bit 
            Log "`nKernel32::LoadLibrary   64bit --> 0x$("{0:X16}" -f $LibHandle.ToInt64())"
            Log "HstiTest2::QueryHSTIdetails  64bit --> 0x$("{0:X16}" -f $FuncHandle.ToInt64())"
        }
        else
        {
            return
        }
        $overallError = New-Object HstiTest3+HstiOverallError
        $providerErrorDupleCount = New-Object int
        $blobByteSize = New-Object int
        $hr = [HstiTest3]::QueryHSTIdetails([ref] $overallError, $null, [ref] $providerErrorDupleCount, $null, [ref] $blobByteSize)

        [byte[]]$blob = New-Object byte[] $blobByteSize
        [HstiTest3+HstiProviderErrorDuple[]]$providerErrors = New-Object HstiTest3+HstiProviderErrorDuple[] $providerErrorDupleCount 
        $hr = [HstiTest3]::QueryHSTIdetails([ref] $overallError, $providerErrors, [ref] $providerErrorDupleCount, $blob, [ref] $blobByteSize)
        $string = $null
        $blob | foreach { $string = $string + $_.ToString("X2")+"," }

        $hstiStatus = New-Object bool
        $hr = [HstiTest3]::QueryHSTI([ref] $hstiStatus)

        LogAndConsole "HSTI Duple Count: $providerErrorDupleCount"
        LogAndConsole "HSTI Blob size: $blobByteSize"
        LogAndConsole "String: $string"
        LogAndConsole "HSTIStatus: $hstiStatus"
        if(($blobByteSize -gt 512) -and ($providerErrorDupleCount -gt 0) -and $hstiStatus)
        {
            LogAndConsoleSuccess "HSTI validation successful"
        }
        elseif(($providerErrorDupleCount -eq 0) -or ($blobByteSize -le 512))
        {
            LogAndConsoleWarning "HSTI is absent"
            $DGVerifyWarn.AppendLine("HSTI is absent") | Out-Null
        }
        else
        {
            $ErrorMessage = "HSTI validation failed"
            if($HLK)
            {
                LogAndConsoleError $ErrorMessage
                $DGVerifyCrit.AppendLine($ErrorMessage) | Out-Null
            }
            else 
            {
                LogAndConsoleWarning $ErrorMessage
                $DGVerifyWarn.AppendLine("HSTI is absent") | Out-Null
            }
        }

    }
    catch 
    {
        LogAndConsoleError $_.Exception.Message 
        LogAndConsoleError "Instantiate-HSTI failed"
    }
}


function CheckDGRunning($_val)
{
    $DGObj = Get-CimInstance -classname Win32_DeviceGuard -namespace root\Microsoft\Windows\DeviceGuard
    for($i=0; $i -lt $DGObj.SecurityServicesRunning.length; $i++)
    {
        if($DGObj.SecurityServicesRunning[$i] -eq $_val)
        {
            return 1
        }

    }
    return 0
}

function CheckDGFeatures($_val)
{
    $DGObj = Get-CimInstance -classname Win32_DeviceGuard -namespace root\Microsoft\Windows\DeviceGuard
    Log "DG_obj $DG_obj"
    Log "DG_obj.AvailableSecurityProperties.length $DG_obj.AvailableSecurityProperties.length"
    for($i=0; $i -lt $DGObj.AvailableSecurityProperties.length; $i++)
    {
        if($DGObj.AvailableSecurityProperties[$i] -eq $_val)
        {
            return 1
        }

    }
    return 0
}

function PrintConfigCIDetails($_ConfigCIState)
{
    $_ConfigCIRunning = "Config-CI is enabled and running."
    $_ConfigCIDisabled = "Config-CI is not running."
    $_ConfigCIMode = "Not Enabled"
    switch ($_ConfigCIState)
    {
        0 { $_ConfigCIMode = "Not Enabled" }
        1 { $_ConfigCIMode = "Audit mode" }
        2 { $_ConfigCIMode = "Enforced mode" }
        default { $_ConfigCIMode = "Not Enabled" }
    }

    if($_ConfigCIState -ge 1)
    {
        LogAndConsoleSuccess "$_ConfigCIRunning ($_ConfigCIMode)"
    }
    else
    {
        LogAndConsoleWarning "$_ConfigCIDisabled ($_ConfigCIMode)"
    }
}

function PrintHVCIDetails($_HVCIState)
{
    $_HvciRunning = "HVCI is enabled and running."
    $_HvciDisabled = "HVCI is not running."

    if($_HVCIState)
    {
        LogAndConsoleSuccess $_HvciRunning
    }
    else
    {
        LogAndConsoleWarning $_HvciDisabled
    }
}

function PrintCGDetails ($_CGState)
{
    $_CGRunning = "Credential-Guard is enabled and running."
    $_CGDisabled = "Credential-Guard is not running."

    if($_CGState)
    {
        LogAndConsoleSuccess $_CGRunning
    }
    else
    {
        LogAndConsoleWarning $_CGDisabled
    }
}

if(![IO.Directory]::Exists($path))
{
    New-Item -ItemType directory -Path $path
}
else
{
    #Do Nothing!!
}

function IsRedstone
{
    $_osVersion = [environment]::OSVersion.Version
    Log $_osVersion
    #Check if build Major is Windows 10
    if($_osVersion.Major -lt 10)
    {
        return 0
    }
    #Check if the build is post Threshold2 (1511 release) => Redstone
    if($_osVersion.Build -gt 10586)
    {
        return 1
    }
    #default return False
    return 0
}

function ExecuteCommandAndLog($_cmd)
{
    try
    {
        Log "Executing: $_cmd"
        $CmdOutput = Invoke-Expression $_cmd | Out-String
        Log "Output: $CmdOutput"
    }
    catch 
    {
        Log "Exception while exectuing $_cmd"
        Log $_.Exception.Message 
    }


}

function PrintRebootWarning
{
    LogAndConsoleWarning "Please reboot the machine, for settings to be applied."
}

function AutoRebootHelper
{
    if($AutoReboot)
    {
        LogAndConsole "PC will restart in 30 seconds"
        ExecuteCommandAndLog 'shutdown /r /t 30'
    }
    else
    {
        PrintRebootWarning
    }

}

function VerifierReset
{
    $verifier_state = verifier /query | Out-String
    if(!$verifier_state.ToString().Contains("No drivers are currently verified."))
    {
        ExecuteCommandAndLog 'verifier.exe /reset'
    }
    AutoRebootHelper
}

function PrintHardwareReq
{
    LogAndConsole "###########################################################################"
    LogAndConsole "OS and Hardware requirements for enabling Device Guard and Credential Guard"
    LogAndConsole " 1. OS SKUs: Available only on these OS Skus - Enterprise, Server, Education, Enterprise IoT, Pro, and Home"
    LogAndConsole " 2. Hardware: Recent hardware that supports virtualization extension with SLAT"
    LogAndConsole "To learn more please visit: https://aka.ms/dgwhcr"
    LogAndConsole "########################################################################### `n"
}

function CheckDriverCompat
{
    $_HVCIState = CheckDGRunning(2)
    if($_HVCIState)
    {
        LogAndConsoleWarning "HVCI is already enabled on this machine, driver compat list might not be complete."
        LogAndConsoleWarning "Please disable HVCI and run the script again..."
    }
    $verifier_state = verifier /query | Out-String
    if($verifier_state.ToString().Contains("No drivers are currently verified."))
    {
        LogAndConsole "Enabling Driver verifier"
        verifier.exe /flags 0x02000000 /all /log.code_integrity

        LogAndConsole "Enabling Driver Verifier and Rebooting system"
        Log $verifier_state 
        LogAndConsole "Please re-execute this script after reboot...."
        if($AutoReboot)
        {
            LogAndConsole "PC will restart in 30 seconds"
            ExecuteCommandAndLog 'shutdown /r /t 30'
        }
        else
        {
            LogAndConsole "Please reboot manually and run the script again...."
        }
        exit
    }
    else
    {
        LogAndConsole "Driver verifier already enabled"
        Log $verifier_state 
        ListDrivers($verifier_state.Trim().ToLowerInvariant())
    }
}
function IsDomainController
{
    $_isDC = 0
    $CompConfig = Get-WmiObject Win32_ComputerSystem
    foreach ($ObjItem in $CompConfig) 
    {
        $Role = $ObjItem.DomainRole
        Log "Role=$Role"
        Switch ($Role) 
        {
            0 { Log "Standalone Workstation" }
            1 { Log "Member Workstation" }
            2 { Log "Standalone Server" }
            3 { Log "Member Server" }
            4 
            {
                Log "Backup Domain Controller"
                $_isDC=1
                break
            }
            5 
            {
                Log "Primary Domain Controller"
                $_isDC=1
                break
            }
            default { Log "Unknown Domain Role" }
        }
    }
    return $_isDC
}

function CheckOSSKU
{
    $osname = $((gwmi win32_operatingsystem).Name).ToLower()
    $_SKUSupported = 0
    Log "OSNAME:$osname"
    $SKUarray = @("Enterprise", "Education", "IoT", "Windows Server", "Pro", "Home")
    $HLKAllowed = @("microsoft windows 10 pro")
    foreach ($SKUent in $SKUarray) 
    {
        if($osname.ToString().Contains($SKUent.ToLower()))
        {
            $_SKUSupported = 1
            break
        }
    }

    # For running HLK tests only, professional SKU's are marked as supported.
    if($HLK)
    {
        if($osname.ToString().Contains($HLKAllowed.ToLower()))
        {
            $_SKUSupported = 1
        }
    }
    $_isDomainController = IsDomainController
    if($_SKUSupported)
    {
        LogAndConsoleSuccess "This PC edition is Supported for DeviceGuard";
        if(($_isDomainController -eq 1) -and !$HVCI -and !$DG)
        {
            LogAndConsoleError "This PC is configured as a Domain Controller, Credential Guard is not supported on DC."
        }
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "OSSKU" /t REG_DWORD /d 2 /f '
    }
    else 
    {
        LogAndConsoleError "This PC edition is Unsupported for Device Guard"
        $DGVerifyCrit.AppendLine("OS SKU unsupported") | Out-Null
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "OSSKU" /t REG_DWORD /d 0 /f '
    }
}

function CheckOSArchitecture
{
    $OSArch = $(gwmi win32_operatingsystem).OSArchitecture
    Log $OSArch 
    if($OSArch.Contains("64-bit"))
    {
        LogAndConsoleSuccess "64 bit arch....."
    }
    elseif($OSArch.Contains("32-bit"))
    {
        LogAndConsoleError "32 bit arch...."   
        $DGVerifyCrit.AppendLine("32 Bit OS, OS Architecture failure..") | Out-Null
    }
    else
    {
        LogAndConsoleError "Unknown architecture"
        $DGVerifyCrit.AppendLine("Unknown OS, OS Architecture failure..") | Out-Null
    }
}

function CheckSecureBootState
{
    $_secureBoot = Confirm-SecureBootUEFI
    Log $_secureBoot
    if($_secureBoot)
    {
        LogAndConsoleSuccess "Secure Boot is present"
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "SecureBoot" /t REG_DWORD /d 2 /f '
    }
    else
    {
        LogAndConsoleError "Secure Boot is absent / not enabled."
        LogAndConsoleError "If Secure Boot is supported on the system, enable Secure Boot in the BIOS and run the script again."
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "SecureBoot" /t REG_DWORD /d 0 /f '
        $DGVerifyCrit.AppendLine("Secure boot validation failed.") | Out-Null
    }
}

function CheckVirtualization
{
    $_vmmExtension = $(gwmi -Class Win32_processor).VMMonitorModeExtensions
    $_vmFirmwareExtension = $(gwmi -Class Win32_processor).VirtualizationFirmwareEnabled
    $_vmHyperVPresent =  (gcim -Class Win32_ComputerSystem).HypervisorPresent
    Log "VMMonitorModeExtensions $_vmmExtension"
    Log "VirtualizationFirmwareEnabled $_vmFirmwareExtension"
    Log "HyperVisorPresent $_vmHyperVPresent"

    #success if either processor supports and enabled or if hyper-v is present
    if(($_vmmExtension -and $_vmFirmwareExtension) -or $_vmHyperVPresent )
    {
        LogAndConsoleSuccess "Virtualization firmware check passed"
         ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "Virtualization" /t REG_DWORD /d 2 /f '
    }
    else
    {
        LogAndConsoleError "Virtualization firmware check failed."
        LogAndConsoleError "If Virtualization extensions are supported on the system, enable hardware virtualization (Intel Virtualization Technology, Intel VT-x, Virtualization Extensions, or similar) in the BIOS and run the script again."
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "Virtualization" /t REG_DWORD /d 0 /f '
        $DGVerifyCrit.AppendLine("Virtualization firmware check failed.") | Out-Null
    }
}

function CheckTPM
{
    $TPMLockout = $(get-tpm).LockoutCount

    if($TPMLockout)
    {

        if($TPMLockout.ToString().Contains("Not Supported for TPM 1.2"))
        {
            if($HLK)
            {
                LogAndConsoleSuccess "TPM 1.2 is present."
            }
            else
            {
                $WarningMsg = "TPM 1.2 is Present. TPM 2.0 is Preferred."
                LogAndConsoleWarning $WarningMsg
                $DGVerifyWarn.AppendLine($WarningMsg) | Out-Null
            }
        }
        else
        {
            LogAndConsoleSuccess "TPM 2.0 is present."
        }
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "TPM" /t REG_DWORD /d 2 /f '
    }
    else
    {
        $WarningMsg = "TPM is absent or not ready for use"
        if($HLK)
        {
            LogAndConsoleError $WarningMsg
            $DGVerifyCrit.AppendLine($WarningMsg) | Out-Null
        }
        else
        {
            LogAndConsoleWarning $WarningMsg
            $DGVerifyWarn.AppendLine($WarningMsg) | Out-Null
        }
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "TPM" /t REG_DWORD /d 0 /f '
    }
}

function CheckSecureMOR
{
    $isSecureMOR = CheckDGFeatures(4)
    Log "isSecureMOR= $isSecureMOR " 
    if($isSecureMOR -eq 1)
    {
        LogAndConsoleSuccess "Secure MOR is available"
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "SecureMOR" /t REG_DWORD /d 2 /f '
    }
    else
    {
        $WarningMsg = "Secure MOR is absent"
        if($HLK)
        {
            LogAndConsoleError $WarningMsg
            $DGVerifyCrit.AppendLine($WarningMsg) | Out-Null
        }
        else
        {
            LogAndConsoleWarning $WarningMsg
            $DGVerifyWarn.AppendLine($WarningMsg) | Out-Null
        }
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "SecureMOR" /t REG_DWORD /d 0 /f '
    }
}

function CheckNXProtection
{
    $isNXProtected = CheckDGFeatures(5)
    Log "isNXProtected= $isNXProtected " 
    if($isNXProtected -eq 1)
    {
        LogAndConsoleSuccess "NX Protector is available"
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "UEFINX" /t REG_DWORD /d 2 /f '
    }
    else
    {
        LogAndConsoleWarning "NX Protector is absent"
        $DGVerifyWarn.AppendLine("NX Protector is absent") | Out-Null
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "UEFINX" /t REG_DWORD /d 0 /f '
    }
}

function CheckSMMProtection
{
    $isSMMMitigated = CheckDGFeatures(6)
    Log "isSMMMitigated= $isSMMMitigated " 
    if($isSMMMitigated -eq 1)
    {
        LogAndConsoleSuccess "SMM Mitigation is available"
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "SMMProtections" /t REG_DWORD /d 2 /f '
    }
    else
    {
        LogAndConsoleWarning "SMM Mitigation is absent"
        $DGVerifyWarn.AppendLine("SMM Mitigation is absent") | Out-Null
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "SMMProtections" /t REG_DWORD /d 0 /f '
    }
}

function CheckHSTI
{
    LogAndConsole "Copying HSTITest.dll"
    try 
    {
        $HSTITest_Decoded = [System.Convert]::FromBase64String($HSTITest_Encoded)
        [System.IO.File]::WriteAllBytes("$env:windir\System32\hstitest.dll",$HSTITest_Decoded)

    }
    catch 
    {
        LogAndConsole $_.Exception.Message 
        LogAndConsole "Copying and loading HSTITest.dll failed"
    }

    Instantiate-Kernel32
    Instantiate-HSTI
}

function PrintToolVersion
{
    LogAndConsole "###########################################################################"
    LogAndConsole "Readiness Tool Version 3.4 Release. `nTool to check if your device is capable to run Device Guard and Credential Guard."
    LogAndConsole "###########################################################################"

}

PrintToolVersion

if(!($Ready) -and !($Capable) -and !($Enable) -and !($Disable) -and !($Clear) -and !($ResetVerifier))
{
    #Print Usage if none of the options are specified
    LogAndConsoleWarning "How to read the output:"
    LogAndConsoleWarning " 1. Red Errors: Basic things are missing that will prevent enabling and using DG/CG"
    LogAndConsoleWarning " 2. Yellow Warnings: This device can be used to enable and use DG/CG, but additional security benefits will be absent. To learn more please go through: https://aka.ms/dgwhcr"
    LogAndConsoleWarning " 3. Green Messages: This device is fully compliant with DG/CG requirements`n"
    LogAndConsoleWarning "###########################################################################"
    LogAndConsoleWarning "Hardware requirements for enabling Device Guard and Credential Guard"
    LogAndConsoleWarning " 1. Hardware: Recent hardware that supports virtualization extension with SLAT"
    LogAndConsoleWarning "########################################################################### `n"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -[Capable/Ready/Enable/Disable/Clear] -[DG/CG/HVCI] -[AutoReboot] -Path"
    LogAndConsoleWarning "Log file with details is found here: C:\DGLogs `n"

    LogAndConsoleWarning "To Enable DG/CG. If you have a custom SIPolicy.p7b then use the -Path parameter else the hardcoded default policy is used"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Enable OR DG_Readiness.ps1 -Enable -Path <full path to the SIPolicy.p7b> `n"

    LogAndConsoleWarning "To Enable only HVCI"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Enable -HVCI `n"

    LogAndConsoleWarning "To Enable only CG"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Enable -CG `n"

    LogAndConsoleWarning "To Verify if DG/CG is enabled"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Ready `n"

    LogAndConsoleWarning "To Disable DG/CG."
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Disable `n"

    LogAndConsoleWarning "To Verify if DG/CG is disabled"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Ready `n"

    LogAndConsoleWarning "To Verify if this device is DG/CG Capable"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Capable"

    LogAndConsoleWarning "To Verify if this device is HVCI Capable"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -Capable -HVCI"

    LogAndConsoleWarning "To Auto reboot with each option"
    LogAndConsoleWarning "Usage: DG_Readiness.ps1 -[Capable/Enable/Disable] -AutoReboot"
    LogAndConsoleWarning "###########################################################################"
    LogAndConsoleWarning "Readiness Tool with '-capable' is run the following RegKey values are set:"
    LogAndConsoleWarning "HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities"
    LogAndConsoleWarning "CG_Capable"
    LogAndConsoleWarning "DG_Capable"
    LogAndConsoleWarning "HVCI_Capable"
    LogAndConsoleWarning "Value 0 = not possible to enable DG/CG/HVCI on this device"
    LogAndConsoleWarning "Value 1 = not fully compatible but has sufficient firmware/hardware/software features to enable DG/CG/HVCI"
    LogAndConsoleWarning "Value 2 = fully compatible for DG/CG/HVCI"
    LogAndConsoleWarning "########################################################################### `n"
}

$user = [Security.Principal.WindowsIdentity]::GetCurrent();
$TestForAdmin = (New-Object Security.Principal.WindowsPrincipal $user).IsInRole([Security.Principal.WindowsBuiltinRole]::Administrator)  

if(!$TestForAdmin)
{
    LogAndConsoleError "Not an Admin user, pls execute this script as an Admin user exiting..."
    exit
}

$isRunningOnVM = (get-wmiobject win32_computersystem).model
if($isRunningOnVM.Contains("Virtual"))
{
    LogAndConsoleWarning "Running on a Virtual Machine. DG/CG is supported only if both guest VM and host machine are running with Windows 10, version 1703 or later with English localization."
}


<# Check the DG status if enabled or disabled, meaning if the device is ready or not #>
if($Ready)
{
    PrintHardwareReq

    $DGRunning = $(Get-CimInstance -classname Win32_DeviceGuard -namespace root\Microsoft\Windows\DeviceGuard).SecurityServicesRunning
    $_ConfigCIState = $(Get-CimInstance -classname Win32_DeviceGuard -namespace root\Microsoft\Windows\DeviceGuard).CodeIntegrityPolicyEnforcementStatus
    Log "Current DGRunning = $DGRunning, ConfigCI= $_ConfigCIState"
    $_HVCIState = CheckDGRunning(2)
    $_CGState = CheckDGRunning(1)

    if($HVCI)
    {
        Log "_HVCIState: $_HVCIState"
        PrintHVCIDetails $_HVCIState
    }
    elseif($CG)
    {
        Log "_CGState: $_CGState"
        PrintCGDetails $_CGState
        
        if($_CGState)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "CG_Running" /t REG_DWORD /d 1 /f'
        }
        else
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "CG_Running" /t REG_DWORD /d 0 /f'
        }
    }
    elseif($DG)
    {
        Log "_HVCIState: $_HVCIState, _ConfigCIState: $_ConfigCIState" 

        PrintHVCIDetails $_HVCIState
        PrintConfigCIDetails $_ConfigCIState 

        if($_ConfigCIState -and $_HVCIState)
        {
            LogAndConsoleSuccess "HVCI, and Config-CI are enabled and running."
            
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "DG_Running" /t REG_DWORD /d 1 /f'
        }
        else
        {
            LogAndConsoleWarning "Not all services are running."
            
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "DG_Running" /t REG_DWORD /d 0 /f'
        }
    }
    else 
    {
        Log "_CGState: $_CGState, _HVCIState: $_HVCIState, _ConfigCIState: $_ConfigCIState" 
        
        PrintCGDetails $_CGState
        PrintHVCIDetails $_HVCIState
        PrintConfigCIDetails $_ConfigCIState

        if(($DGRunning.Length -ge 2) -and ($_CGState) -and ($_HVCIState) -and ($_ConfigCIState -ge 1))
        {
            LogAndConsoleSuccess "HVCI, Credential-Guard, and Config-CI are enabled and running."
        }
        else
        {
            LogAndConsoleWarning "Not all services are running."
        }
    }
}

<# Enable and Disable #>
if($Enable)
{
    PrintHardwareReq

    LogAndConsole "Enabling Device Guard and Credential Guard"
    LogAndConsole "Setting RegKeys to enable DG/CG"

    ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "EnableVirtualizationBasedSecurity" /t REG_DWORD /d 1 /f'
    #Only SecureBoot is required as part of RequirePlatformSecurityFeatures
    ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "RequirePlatformSecurityFeatures" /t REG_DWORD /d 1 /f'

    $_isRedstone = IsRedstone
    if(!$_isRedstone)
    {
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "Unlocked" /t REG_DWORD /d 1 /f'
    }
    else
    {
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "Locked" /t REG_DWORD /d 0 /f'
    }

    if(!$HVCI -and !$DG)
    {
        # value is 2 for both Th2 and RS1
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\Lsa" /v "LsaCfgFlags" /t REG_DWORD /d 2 /f'
    }
    if(!$CG)
    {
        if(!$_isRedstone)
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "HypervisorEnforcedCodeIntegrity" /t REG_DWORD /d 1 /f'
        }
        else 
        {
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Scenarios\HypervisorEnforcedCodeIntegrity" /v "Enabled" /t REG_DWORD /d 1 /f'
            ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Scenarios\HypervisorEnforcedCodeIntegrity" /v "Locked" /t REG_DWORD /d 0 /f'
        }
    }

    try
    {
        if(!$HVCI -and !$CG)
        {
            if(!$SIPolicyPath) 
            {  
                Log "Writing Decoded SIPolicy.p7b"
                $SIPolicy_Decoded = [System.Convert]::FromBase64String($SIPolicy_Encoded)
                [System.IO.File]::WriteAllBytes("$env:windir\System32\CodeIntegrity\SIPolicy.p7b",$SIPolicy_Decoded)
            }
            else
            {
                LogAndConsole "Copying user provided SIpolicy.p7b"
                $CmdOutput = Copy-Item $SIPolicyPath "$env:windir\System32\CodeIntegrity\SIPolicy.p7b" | Out-String
                Log $CmdOutput
            }
        }
    }
    catch
    {
        LogAndConsole "Writing SIPolicy.p7b file failed"
    }

    LogAndConsole "Enabling Hyper-V and IOMMU"
    $_isRedstone = IsRedstone
    if(!$_isRedstone)
    {
        LogAndConsole "OS Not Redstone, enabling IsolatedUserMode separately"
        #Enable/Disable IOMMU seperately
        ExecuteCommandAndLog 'DISM.EXE /Online /Enable-Feature:IsolatedUserMode /NoRestart'
    }
    $CmdOutput =  DISM.EXE /Online /Enable-Feature:Microsoft-Hyper-V-Hypervisor /All /NoRestart | Out-String
    if(!$CmdOutput.Contains("The operation completed successfully."))
    {
        $CmdOutput =  DISM.EXE /Online /Enable-Feature:Microsoft-Hyper-V-Online /All /NoRestart | Out-String
    }

    Log $CmdOutput
    if($CmdOutput.Contains("The operation completed successfully."))
    {
        LogAndConsoleSuccess "Enabling Hyper-V and IOMMU successful"
        #Reg key for HLK validation of DISM.EXE step
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "HyperVEnabled" /t REG_DWORD /d 1 /f'
    }
    else
    {
        LogAndConsoleWarning "Enabling Hyper-V failed please check the log file"
        #Reg key for HLK validation of DISM.EXE step
        ExecuteCommandAndLog 'REG ADD "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities\" /v "HyperVEnabled" /t REG_DWORD /d 0 /f'
    }
    AutoRebootHelper
}

if($Disable)
{
    LogAndConsole "Disabling Device Guard and Credential Guard"
    LogAndConsole "Deleting RegKeys to disable DG/CG"

    ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "EnableVirtualizationBasedSecurity" /f'
    ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "RequirePlatformSecurityFeatures" /f'

    $_isRedstone = IsRedstone
    if(!$_isRedstone)
    {
        ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "NoLock" /f'
    }
    else
    {
        ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "Locked" /f'
    }

    if(!$CG)
    {
        ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard" /v "HypervisorEnforcedCodeIntegrity" /f'
        if($_isRedstone)
        {
            ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Scenarios\HypervisorEnforcedCodeIntegrity" /f'
        }
    }

    if(!$HVCI -and !$DG)
    {
        ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\Lsa" /v "LsaCfgFlags" /f'
    }

    if(!$HVCI -and !$CG)
    {
        ExecuteCommandAndLog 'del  "$env:windir\System32\CodeIntegrity\SIPolicy.p7b"'
    }

    if(!$HVCI -and !$DG -and !$CG)
    {
        LogAndConsole "Disabling Hyper-V and IOMMU"
        $_isRedstone = IsRedstone
        if(!$_isRedstone)
        {
            LogAndConsole "OS Not Redstone, disabling IsolatedUserMode separately"
            #Enable/Disable IOMMU seperately
            ExecuteCommandAndLog 'DISM.EXE /Online /disable-Feature /FeatureName:IsolatedUserMode /NoRestart'
        }
        $CmdOutput =  DISM.EXE /Online /disable-Feature /FeatureName:Microsoft-Hyper-V-Hypervisor /NoRestart | Out-String
        if(!$CmdOutput.Contains("The operation completed successfully."))
        {
            $CmdOutput =  DISM.EXE /Online /disable-Feature /FeatureName:Microsoft-Hyper-V-Online /NoRestart | Out-String
        }
        Log $CmdOutput
        if($CmdOutput.Contains("The operation completed successfully."))
        {
            LogAndConsoleSuccess "Disabling Hyper-V and IOMMU successful"
        }
        else
        {
            LogAndConsoleWarning "Disabling Hyper-V failed please check the log file"
        }

        #set of commands to run SecConfig.efi to delete UEFI variables if were set in pre OS
        #these steps can be performed even if the UEFI variables were not set - if not set it will lead to No-Op but this can be run in general always 
        #this requires a reboot and accepting the prompt in the Pre-OS which is self explanatory in the message that is displayed in pre-OS
        $FreeDrive = ls function:[s-z]: -n | ?{ !(test-path $_) } | random
        Log "FreeDrive=$FreeDrive"
        ExecuteCommandAndLog 'mountvol $FreeDrive /s'
        $CmdOutput = Copy-Item "$env:windir\System32\SecConfig.efi" $FreeDrive\EFI\Microsoft\Boot\SecConfig.efi -Force | Out-String
        LogAndConsole $CmdOutput
        ExecuteCommandAndLog 'bcdedit /create "{0cb3b571-2f2e-4343-a879-d86a476d7215}" /d DGOptOut /application osloader'
        ExecuteCommandAndLog 'bcdedit /set "{0cb3b571-2f2e-4343-a879-d86a476d7215}" path \EFI\Microsoft\Boot\SecConfig.efi'
        ExecuteCommandAndLog 'bcdedit /set "{bootmgr}" bootsequence "{0cb3b571-2f2e-4343-a879-d86a476d7215}"'
        ExecuteCommandAndLog 'bcdedit /set "{0cb3b571-2f2e-4343-a879-d86a476d7215}" loadoptions DISABLE-LSA-ISO,DISABLE-VBS'
        ExecuteCommandAndLog 'bcdedit /set "{0cb3b571-2f2e-4343-a879-d86a476d7215}" device partition=$FreeDrive'
        ExecuteCommandAndLog 'mountvol $FreeDrive /d'
        #steps complete

    }
    AutoRebootHelper
}

if($Clear)
{
    ExecuteCommandAndLog 'REG DELETE "HKLM\SYSTEM\CurrentControlSet\Control\DeviceGuard\Capabilities" /f'
    VerifierReset
}

if($ResetVerifier)
{
    VerifierReset
}

<# Is machine Device Guard / Cred Guard Capable and Verify #>
if($Capable)
{
    PrintHardwareReq

    LogAndConsole "Checking if the device is DG/CG Capable"

    $_isRedstone = IsRedstone
    if(!$_isRedstone)
    {
        LogAndConsoleWarning "Capable is currently fully supported in Redstone only.."
    }
    $_StepCount = 1
    if(!$CG)
    {   
        LogAndConsole " ====================== Step $_StepCount Driver Compat ====================== "
        $_StepCount++
        CheckDriverCompat
    }

    LogAndConsole " ====================== Step $_StepCount Secure boot present ====================== "
    $_StepCount++
    CheckSecureBootState
    
    if(!$HVCI -and !$DG -and !$CG)
    {   
        #check only if sub-options are absent
        LogAndConsole " ====================== Step $_StepCount MS UEFI HSTI tests ====================== "
        $_StepCount++
        CheckHSTI
    }
    
    LogAndConsole " ====================== Step $_StepCount OS Architecture ====================== "
    $_StepCount++
    CheckOSArchitecture

    LogAndConsole " ====================== Step $_StepCount Supported OS SKU ====================== "
    $_StepCount++
    CheckOSSKU

    LogAndConsole " ====================== Step $_StepCount Virtualization Firmware ====================== "
    $_StepCount++
    CheckVirtualization

    if(!$HVCI -and !$DG)
    {   
        LogAndConsole " ====================== Step $_StepCount TPM version ====================== "
        $_StepCount++
        CheckTPM
        
        LogAndConsole " ====================== Step $_StepCount Secure MOR ====================== "
        $_StepCount++
        CheckSecureMOR
    }

    LogAndConsole " ====================== Step $_StepCount NX Protector ====================== "
    $_StepCount++
    CheckNXProtection
    
    LogAndConsole " ====================== Step $_StepCount SMM Mitigation ====================== "
    $_StepCount++
    CheckSMMProtection
    
    LogAndConsole " ====================== End Check ====================== "

    LogAndConsole " ====================== Summary ====================== "
    ListSummary
    LogAndConsole "To learn more about required hardware and software please visit: https://aka.ms/dgwhcr"
}



# SIG # Begin signature block
# MIIkiAYJKoZIhvcNAQcCoIIkeTCCJHUCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCHxqndWS8H7Xe0
# yxglOZe0uxNr5UZCwWfGVssI1GUNHaCCDYEwggX/MIID56ADAgECAhMzAAABA14l
# HJkfox64AAAAAAEDMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMTgwNzEyMjAwODQ4WhcNMTkwNzI2MjAwODQ4WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQDRlHY25oarNv5p+UZ8i4hQy5Bwf7BVqSQdfjnnBZ8PrHuXss5zCvvUmyRcFrU5
# 3Rt+M2wR/Dsm85iqXVNrqsPsE7jS789Xf8xly69NLjKxVitONAeJ/mkhvT5E+94S
# nYW/fHaGfXKxdpth5opkTEbOttU6jHeTd2chnLZaBl5HhvU80QnKDT3NsumhUHjR
# hIjiATwi/K+WCMxdmcDt66VamJL1yEBOanOv3uN0etNfRpe84mcod5mswQ4xFo8A
# DwH+S15UD8rEZT8K46NG2/YsAzoZvmgFFpzmfzS/p4eNZTkmyWPU78XdvSX+/Sj0
# NIZ5rCrVXzCRO+QUauuxygQjAgMBAAGjggF+MIIBejAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQUR77Ay+GmP/1l1jjyA123r3f3QP8w
# UAYDVR0RBEkwR6RFMEMxKTAnBgNVBAsTIE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1
# ZXJ0byBSaWNvMRYwFAYDVQQFEw0yMzAwMTIrNDM3OTY1MB8GA1UdIwQYMBaAFEhu
# ZOVQBdOCqhc3NyK1bajKdQKVMFQGA1UdHwRNMEswSaBHoEWGQ2h0dHA6Ly93d3cu
# bWljcm9zb2Z0LmNvbS9wa2lvcHMvY3JsL01pY0NvZFNpZ1BDQTIwMTFfMjAxMS0w
# Ny0wOC5jcmwwYQYIKwYBBQUHAQEEVTBTMFEGCCsGAQUFBzAChkVodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NlcnRzL01pY0NvZFNpZ1BDQTIwMTFfMjAx
# MS0wNy0wOC5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAn/XJ
# Uw0/DSbsokTYDdGfY5YGSz8eXMUzo6TDbK8fwAG662XsnjMQD6esW9S9kGEX5zHn
# wya0rPUn00iThoj+EjWRZCLRay07qCwVlCnSN5bmNf8MzsgGFhaeJLHiOfluDnjY
# DBu2KWAndjQkm925l3XLATutghIWIoCJFYS7mFAgsBcmhkmvzn1FFUM0ls+BXBgs
# 1JPyZ6vic8g9o838Mh5gHOmwGzD7LLsHLpaEk0UoVFzNlv2g24HYtjDKQ7HzSMCy
# RhxdXnYqWJ/U7vL0+khMtWGLsIxB6aq4nZD0/2pCD7k+6Q7slPyNgLt44yOneFuy
# bR/5WcF9ttE5yXnggxxgCto9sNHtNr9FB+kbNm7lPTsFA6fUpyUSj+Z2oxOzRVpD
# MYLa2ISuubAfdfX2HX1RETcn6LU1hHH3V6qu+olxyZjSnlpkdr6Mw30VapHxFPTy
# 2TUxuNty+rR1yIibar+YRcdmstf/zpKQdeTr5obSyBvbJ8BblW9Jb1hdaSreU0v4
# 6Mp79mwV+QMZDxGFqk+av6pX3WDG9XEg9FGomsrp0es0Rz11+iLsVT9qGTlrEOla
# P470I3gwsvKmOMs1jaqYWSRAuDpnpAdfoP7YO0kT+wzh7Qttg1DO8H8+4NkI6Iwh
# SkHC3uuOW+4Dwx1ubuZUNWZncnwa6lL2IsRyP64wggd6MIIFYqADAgECAgphDpDS
# AAAAAAADMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
# ZSBBdXRob3JpdHkgMjAxMTAeFw0xMTA3MDgyMDU5MDlaFw0yNjA3MDgyMTA5MDla
# MH4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMT
# H01pY3Jvc29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTEwggIiMA0GCSqGSIb3DQEB
# AQUAA4ICDwAwggIKAoICAQCr8PpyEBwurdhuqoIQTTS68rZYIZ9CGypr6VpQqrgG
# OBoESbp/wwwe3TdrxhLYC/A4wpkGsMg51QEUMULTiQ15ZId+lGAkbK+eSZzpaF7S
# 35tTsgosw6/ZqSuuegmv15ZZymAaBelmdugyUiYSL+erCFDPs0S3XdjELgN1q2jz
# y23zOlyhFvRGuuA4ZKxuZDV4pqBjDy3TQJP4494HDdVceaVJKecNvqATd76UPe/7
# 4ytaEB9NViiienLgEjq3SV7Y7e1DkYPZe7J7hhvZPrGMXeiJT4Qa8qEvWeSQOy2u
# M1jFtz7+MtOzAz2xsq+SOH7SnYAs9U5WkSE1JcM5bmR/U7qcD60ZI4TL9LoDho33
# X/DQUr+MlIe8wCF0JV8YKLbMJyg4JZg5SjbPfLGSrhwjp6lm7GEfauEoSZ1fiOIl
# XdMhSz5SxLVXPyQD8NF6Wy/VI+NwXQ9RRnez+ADhvKwCgl/bwBWzvRvUVUvnOaEP
# 6SNJvBi4RHxF5MHDcnrgcuck379GmcXvwhxX24ON7E1JMKerjt/sW5+v/N2wZuLB
# l4F77dbtS+dJKacTKKanfWeA5opieF+yL4TXV5xcv3coKPHtbcMojyyPQDdPweGF
# RInECUzF1KVDL3SV9274eCBYLBNdYJWaPk8zhNqwiBfenk70lrC8RqBsmNLg1oiM
# CwIDAQABo4IB7TCCAekwEAYJKwYBBAGCNxUBBAMCAQAwHQYDVR0OBBYEFEhuZOVQ
# BdOCqhc3NyK1bajKdQKVMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1Ud
# DwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaAFHItOgIxkEO5FAVO
# 4eqnxzHRI4k0MFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcmwwXgYIKwYBBQUHAQEEUjBQME4GCCsGAQUFBzAChkJodHRwOi8vd3d3Lm1p
# Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dDIwMTFfMjAxMV8wM18y
# Mi5jcnQwgZ8GA1UdIASBlzCBlDCBkQYJKwYBBAGCNy4DMIGDMD8GCCsGAQUFBwIB
# FjNodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2RvY3MvcHJpbWFyeWNw
# cy5odG0wQAYIKwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AcABvAGwAaQBjAHkA
# XwBzAHQAYQB0AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAGfyhqWY
# 4FR5Gi7T2HRnIpsLlhHhY5KZQpZ90nkMkMFlXy4sPvjDctFtg/6+P+gKyju/R6mj
# 82nbY78iNaWXXWWEkH2LRlBV2AySfNIaSxzzPEKLUtCw/WvjPgcuKZvmPRul1LUd
# d5Q54ulkyUQ9eHoj8xN9ppB0g430yyYCRirCihC7pKkFDJvtaPpoLpWgKj8qa1hJ
# Yx8JaW5amJbkg/TAj/NGK978O9C9Ne9uJa7lryft0N3zDq+ZKJeYTQ49C/IIidYf
# wzIY4vDFLc5bnrRJOQrGCsLGra7lstnbFYhRRVg4MnEnGn+x9Cf43iw6IGmYslmJ
# aG5vp7d0w0AFBqYBKig+gj8TTWYLwLNN9eGPfxxvFX1Fp3blQCplo8NdUmKGwx1j
# NpeG39rz+PIWoZon4c2ll9DuXWNB41sHnIc+BncG0QaxdR8UvmFhtfDcxhsEvt9B
# xw4o7t5lL+yX9qFcltgA1qFGvVnzl6UJS0gQmYAf0AApxbGbpT9Fdx41xtKiop96
# eiL6SJUfq/tHI4D1nvi/a7dLl+LrdXga7Oo3mXkYS//WsyNodeav+vyL6wuA6mk7
# r/ww7QRMjt/fdW1jkT3RnVZOT7+AVyKheBEyIXrvQQqxP/uozKRdwaGIm1dxVk5I
# RcBCyZt2WwqASGv9eZ/BvW1taslScxMNelDNMYIWXTCCFlkCAQEwgZUwfjELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEoMCYGA1UEAxMfTWljcm9z
# b2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAxMQITMwAAAQNeJRyZH6MeuAAAAAABAzAN
# BglghkgBZQMEAgEFAKCB3jAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg7JuFbMny
# prWUtQLsUQtPig0r2O7J0E6gX1e2E8EVS74wcgYKKwYBBAGCNwIBDDFkMGKgJIAi
# AEQARwBfAFIAZQBhAGQAaQBuAGUAcwBzAF8AdgAzAC4ANqE6gDhcXHdpbnNlY3Rc
# c2NyYXRjaFxhbmRyZXdkYVxkZ3JlYWRpbmVzc1xkZ3JlYWRpbmVzc192My42IDAN
# BgkqhkiG9w0BAQEFAASCAQAGOszu1Q5CcLNirOE4ii1niIXKYuNI1TwR75E2Nz5N
# /2R9A8Y7nBA6spI3CU789yMxeXcnL+miNVvGfJ9/TZbEJOHjpdI9/pcAbi2hWgS1
# 4IMOKlSK5BydA5TiWIEeUcUl+T9VF07RJ1bo3FcJjnlqa3CMOUeWRQTr0e1PwVrW
# B/fOpEVJMH+1MmPhXzGORPhzAoifT/2RebgK89bi41bpZQoqiuUwlRecGC6DJuVq
# U1Ozu1WkKlQqimzWtcNfk87IPrDnSfOdfSYBS9glAFZPEjNjUCuaYhme7f3Neudp
# 40e/7FjmKk9xDEyUs8rSHHR0baT6Q3Stb3uP2fh8rP55oYITtzCCE7MGCisGAQQB
# gjcDAwExghOjMIITnwYJKoZIhvcNAQcCoIITkDCCE4wCAQMxDzANBglghkgBZQME
# AgEFADCCAVgGCyqGSIb3DQEJEAEEoIIBRwSCAUMwggE/AgEBBgorBgEEAYRZCgMB
# MDEwDQYJYIZIAWUDBAIBBQAEIPLbteV6DqdVXi1De0Js60tbDSgFRatUr2vRlQHn
# NARGAgZbrQRCU6gYEzIwMTgxMDE5MjIxNzQyLjY0NlowBwIBAYACAfSggdSkgdEw
# gc4xCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
# ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsT
# IE1pY3Jvc29mdCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFs
# ZXMgVFNTIEVTTjo3MjhELUM0NUYtRjlFQjElMCMGA1UEAxMcTWljcm9zb2Z0IFRp
# bWUtU3RhbXAgU2VydmljZaCCDx8wggT1MIID3aADAgECAhMzAAAA09CUVp0OvYMG
# AAAAAADTMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpX
# YXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQg
# Q29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAy
# MDEwMB4XDTE4MDgyMzIwMjY0MFoXDTE5MTEyMzIwMjY0MFowgc4xCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYD
# VQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29mdCBP
# cGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVTTjo3
# MjhELUM0NUYtRjlFQjElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
# dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAK7ynC6AF22joS/v
# TPZsIG82oovZ8kXNQcF6/17dZtRllU6pCGV8zMxSQOXTWD2MZRJ/OqfHUSYCNTPa
# knetNsrZhstlFNT09QBjjeVXayDG/aI8JPy91P5riOAFk/gvjnQCdcoV65OBF286
# bs2lgUa6rc2qKHwDVpR1w+2jXrS8Jtz6omUgfB7CMpw1ZwMeQ/+Fb43EAIxeNXB5
# uq/ZYPDA+iMitkdhrjQJgPKKQqhPiYcz3KdrAk34V6y/zUw8FuJ9Zi89actfoS0e
# AdSdWYDATi6oIiPAioWYQuwx6ZY+e5U8HcjGiA1bg9pnufqcnVLzInBxr8DVp1im
# mAhtkfUCAwEAAaOCARswggEXMB0GA1UdDgQWBBQoUcoPr2oQO5sHaVpYVKDsatRn
# eDAfBgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVtVTBWBgNVHR8ETzBNMEug
# SaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
# aWNUaW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUHAQEETjBMMEoGCCsG
# AQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Rp
# bVN0YVBDQV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoG
# CCsGAQUFBwMIMA0GCSqGSIb3DQEBCwUAA4IBAQA9YvD9FBa0sIj/Q8252GXwW0qQ
# aEm/oZXTh4eI6htIKASVxX8y1g4IVeD6O8YyXdBlzQUgr76B70pDgqyynwmJK6KB
# pg2bf6KOeHImc4pmofFc9EhYLZgXPXwqHJY1Rgwt4X1kCNNK6PTGeFlJproYry38
# a8AuUm0oLJpf46TLC4wQv89vfyEhBed/Wv95Ro5fqn/tAQc8S/c0eq1CAdkMDzsJ
# q7lZmiEAMaVF0vKrcRvtVu7T5BZcTmP6bHNtzcDxnn7rB6TUgSREnWP5Di46Z9P6
# 0XraNff0Ttit5Msy8ivsrcEa2CIxUgscbYDxAaWR8Ghb/rTVIEEWYBAVrF9vMIIG
# cTCCBFmgAwIBAgIKYQmBKgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0
# IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAwNzAxMjEzNjU1
# WhcNMjUwNzAxMjE0NjU1WjB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
# Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
# cmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCC
# ASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8E5f1+n9p
# lGt0VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5mK1vwFVMnBDEfQRsalR3OCROOfGEw
# WbEwRA/xYIiEVEMM1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEbyWEeG
# MoQedGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC732H8RsEnHSRnEnIaIYqvS2SJ
# UGKxXf13Hz3wV3WsvYpCTUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw
# 2k4GkbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN/LzAyURdXhacAQVPIk0C
# AwEAAaOCAeYwggHiMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ
# 80N7fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
# BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2U
# kFvXzpoYxDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
# b20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcmww
# WgYIKwYBBQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29m
# dC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1dF8yMDEwLTA2LTIzLmNydDCBoAYD
# VR0gAQH/BIGVMIGSMIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0dHA6
# Ly93d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9DUFMvZGVmYXVsdC5odG0wQAYI
# KwYBBQUHAgIwNB4yIB0ATABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0
# AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIBAAfmiFEN4sbgmD+BcQM9
# naOhIW+z66bM9TG+zwXiqf76V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtR
# gkQS+7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDIr79/xn/yN31aPxzy
# mXlKkVIArzgPF/UveYFl2am1a+THzvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCf
# Mkon/VWvL/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jnsGUpxY517IW3D
# nKOiPPp/fZZqkHimbdLhnPkd/DjYlPTGpQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs
# 9/S/fmNZJQ96LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/DhO3EJ3110
# mCIIYdqwUB5vvfHhAN/nMQekkzr3ZUd46PioSKv33nJ+YWtvd6mBy6cJrDm77MbL
# 2IK0cs0d9LiFAR6A+xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1QnIffI
# rE7aKLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0MkvfY3v1mYovG8chr1m1rtxE
# PJdQcdeh0sVV42neV8HR3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc
# 1bN+NR4Iuto229Nfj950iEkSoYIDrTCCApUCAQEwgf6hgdSkgdEwgc4xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKTAnBgNVBAsTIE1pY3Jvc29m
# dCBPcGVyYXRpb25zIFB1ZXJ0byBSaWNvMSYwJAYDVQQLEx1UaGFsZXMgVFNTIEVT
# Tjo3MjhELUM0NUYtRjlFQjElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAg
# U2VydmljZaIlCgEBMAkGBSsOAwIaBQADFQBnQlpxrvQi2lklNcOL1G5qmRJdZ6CB
# 3jCB26SB2DCB1TELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
# BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEp
# MCcGA1UECxMgTWljcm9zb2Z0IE9wZXJhdGlvbnMgUHVlcnRvIFJpY28xJzAlBgNV
# BAsTHm5DaXBoZXIgTlRTIEVTTjo1N0Y2LUMxRTAtNTU0QzErMCkGA1UEAxMiTWlj
# cm9zb2Z0IFRpbWUgU291cmNlIE1hc3RlciBDbG9jazANBgkqhkiG9w0BAQUFAAIF
# AN90i2AwIhgPMjAxODEwMjAwMDU3MDRaGA8yMDE4MTAyMTAwNTcwNFowdDA6Bgor
# BgEEAYRZCgQBMSwwKjAKAgUA33SLYAIBADAHAgEAAgIbuDAHAgEAAgIVjTAKAgUA
# 33Xc4AIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEEAYRZCgMBoAowCAIBAAID
# FuNgoQowCAIBAAIDB6EgMA0GCSqGSIb3DQEBBQUAA4IBAQBN4fOQLT2kR2D+iliM
# qCyd/OohEKUylqoyzmbzHUcPoCOffrEfwv62lLeZGdZt+IvIcPK1SflWPZIhGkl1
# aCiGf1R505R28926nkarMfFt6C7z4ZrpoEyt3N4YGtl1183lVrRh2xLAh4iPdisc
# bumnWk/n/SPjD5VYBulF6G/TQ0AeVH/d3YjH/DqSZ2W19kvLQEGaTFtGPsUr27zt
# V3O1Wv8Tmyz61GjZYQYFfOrNkemP6aq9JN4nls3np+dcjRUCMZMM7hSsIAirZw4v
# LHR/0b2LbjCZLJyql8SVseNMta3SChDy66eU5VxCK7V2WM2odfVkftCA5GoZYKxj
# YXf2MYIC9TCCAvECAQEwgZMwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIwMTAC
# EzMAAADT0JRWnQ69gwYAAAAAANMwDQYJYIZIAWUDBAIBBQCgggEyMBoGCSqGSIb3
# DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQxIgQgGjZ7SYiSQ2SycmjH
# mRU95/+zeeGqS6SRp9VT53z7lqIwgeIGCyqGSIb3DQEJEAIMMYHSMIHPMIHMMIGx
# BBRnQlpxrvQi2lklNcOL1G5qmRJdZzCBmDCBgKR+MHwxCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVN
# aWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0
# YW1wIFBDQSAyMDEwAhMzAAAA09CUVp0OvYMGAAAAAADTMBYEFFy/e6WhhXZMooPh
# kcyhgus6M/jaMA0GCSqGSIb3DQEBCwUABIIBAJaEchohlJu1U88lEtvq9B34RuR+
# GcmQ+wI6es1eEc+DSbh+glwks1I/yb8KHh6PPu0PiGuVmEH760Bt6JxmgD16yxF0
# 9rzwnq5RkxrXQxxGVfvBBio5PsFOahIv2qqENqUb62ikL2sPxZpHdNe7JzcJVvqy
# eOr9YbK7w75Jl2gbdzar4loQVLLD9bojPaPciSztqRIUsVuiqj2Dq1Vvpujpc8ny
# MZAsWZ0Y5u2lQaVd6z150snjbOMqedGa32j/khlp0sWaOei3kS+tYddPIIAvTKso
# ScY1cd1u4uFUxk6uJEasG1IrsSmsgqowwQyjOv0mFEBBtxdWxfX5ewzysyg=
# SIG # End signature block
